/***************************************************************************
 *
 * Copyright (C) 2001 International Business Machines
 * All rights reserved.
 *
 * This file is part of the GPFS mmfslinux kernel module.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 *
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer. 
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution. 
 *  3. The name of the author may not be used to endorse or promote products 
 *     derived from this software without specific prior written
 *     permission. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *************************************************************************** */
/* Code for program to dump kernel memory that needs to include kernel
   header files. */

/*
 * Contents:
 *   KernInit
 *   GenericGet
 *   GetDentry
 *   FreeDentry
 *   GetInode
 *   GetSuperBlock
 *   GetPage
 *   GetAddressSpace
 *   GetVMAreaStruct
 *   GetMMStruct
 *   GetTaskStruct
 *   GetTaskStructByPID
 *   P_HEADER
 *   PrintDentry
 *   zonePrintInode
 *   PrintSuperBlock
 *   PrintPage
 *   PrintAddressSpace
 *   PrintVMAreaStruct
 *   PrintMMStruct
 *   PrintTaskStruct
 *   VirtToReal
 *
 * $Id: kdump-kern.c,v 1.12.2.2 2002/07/17 14:21:00 gjertsen Exp $
 *
 * $Log: kdump-kern.c,v $
 * Revision 1.12.2.2  2002/07/17 14:21:00  gjertsen
 * Changes for recent kernel version updates with RH 7.2 and 7.3.
 *
 * Revision 1.12.2.1  2002/05/21 21:44:58  dcraft
 * Pull GPFS 1.2.1 up to kernel 2.4.18.
 * mmfsfuncs.Linux must be distributed with /usr/lpp/mmfs/src
 * on developerworks.
 *
 * Revision 1.12  2001/12/04 15:19:17  dcraft
 * Problems with kxMapPrivate on later kernel levels.  Ifdef it out
 * while I figure out whats wrong.
 *
 * Revision 1.11  2001/10/09 17:45:35  dcraft
 * Fixes for running on 2.4.9-ac kernel series. (behind ifdefs)
 *
 * Revision 1.10  2001/10/05 17:48:24  gjertsen
 * Account for kernel vesion 2040312.
 *
 * Revision 1.9  2001/10/03 14:46:13  dcraft
 * First attempt to bring us up to 2.4.9 and 2.4.10
 *
 * Revision 1.8  2001/10/01 14:43:29  dcraft
 * Allow compiling portability layer for 2.4.7-10
 *
 * Revision 1.7  2001/09/28 20:45:48  wyllie
 * Add kdump commands to give formatted dumps of the Linux task_struct and
 * various virtual memory structures.
 *
 * Revision 1.6  2001/08/09 21:11:20  dcraft
 * Modifications to allow running on latest Redhat 7.1 update
 * Kernel version 2.4.3-12.
 * Requires checkout of new site.mcr.proto
 *
 * Revision 1.5  2001/07/13 19:54:45  wyllie
 * Get PAGE_OFFSET by asking the mmfslinux_... kernel module rather than
 * compiling the constant into proprietary code.  Rename constants needed to
 * determine the GPFS memory map and export them to non-proprietary files.
 *
 * Revision 1.4  2001/07/09 15:56:39  wyllie
 * Partial support for running on kernel version 2.4.6
 *
 * Revision 1.3  2001/06/14 18:14:09  gjertsen
 * Initial changes for IA64 beta RH 7.1 with 2.4.3-3 kernel. Get GPFS_PRINTF
 * working again.
 *
 * Revision 1.2  2001/04/02 22:10:02  wyllie
 * Fix low kernel address; pages below C0100000 get freed during kernel init.
 *
 * Revision 1.1  2001/03/16 21:07:34  wyllie
 * Program to dump kernel memory by address or symbol on a running system.
 * Can also get formatted dumps of some kernel data structures.
 *
 */

#include <Shark-gpl.h>
#include <linux/vfs.h>
#include <cxiSystem.h>

#include <linux/config.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/spinlock.h>
#include <linux/smp_lock.h>
#include <linux/dcache.h>
#include <linux/mount.h>
#include <linux/fs.h>

#include <kdump.h>
#include <kdump-kern.h>

#include <stdio.h>
#include <assert.h>

/* Address bounds for dcache hash table */
unsigned long dentry_hashtable;
unsigned int d_hash_mask;
unsigned long dentry_hashtable_limit;

/* Address of dentry_unused list header */
unsigned long dentry_unused = 0x100;

/* Address bounds for inode hash table */
unsigned long inode_hashtable;
unsigned int i_hash_mask;
unsigned long inode_hashtable_limit;

/* Addresses of inode_in_use and inode_unused list headers */
unsigned long inode_in_use = 0x101;
unsigned long inode_unused = 0x102;

/* Address of header of the list of all super_blocks */
unsigned long super_blocks_addr = 0x103;

/* Address of a task_struct on the list of all tasks */
unsigned long TaskAddress = 0x104;

/* Address of the struct page of the first page in memory */
unsigned long MemMapAddr;


/* Routine to initialize stuff needing access to kernel declarations */
void KernInit()
{
  struct Symbol* sP;
  int rc;
  struct vm_struct * vmlist;
  struct vm_struct v;
  struct vmStruct* aListP;
  struct vmStruct** prevPP;
  unsigned long highMem;
  unsigned long endAddr;

  /* Set bounds of valid kernel memory */
  sP = LookupSymbolByName("_stext");
  assert(sP != 0);
  LowKernelAddr = sP->addr;
#ifdef GPFS_ARCH_I386
  LowKernelAddr = PAGE_OFFSET;
#endif
  rc = GetSymbolValue("high_memory", &highMem, sizeof(HighKernelAddr));
  assert(rc == 0);
  sP = LookupSymbolByName("_end");
  assert(sP != 0);
  endAddr = sP->addr;
  if (endAddr > highMem)
    HighKernelAddr = endAddr;
  else
    HighKernelAddr = highMem;

  /* Traverse vm_struct list in kernel starting at vmlist and build
     corresponding list of vmStructs describing valid kernel addresses */
  rc = GetSymbolValue("vmlist", &vmlist, sizeof(vmlist));
  assert(rc == 0);
  vmListHeadP = NULL;
  prevPP = &vmListHeadP;
  while (vmlist != NULL)
  {
    rc = ReadKernel((unsigned long)vmlist, &v, sizeof(v));
    assert(rc == 0);
    aListP = (struct vmStruct*) kMalloc(sizeof(struct vmStruct));
    assert(aListP != NULL);
    DBG(printf("vm_struct addr 0x%lX size %d\n", v.addr, v.size));
    aListP->startAddr = (unsigned long) v.addr;
    aListP->areaLen = (unsigned long) v.size - PAGE_SIZE;
    aListP->nextAreaP = NULL;
    *prevPP = aListP;
    prevPP = &aListP->nextAreaP;
    vmlist = v.next;
  }

  /* Compute bounds of dcache hash table */
  rc = GetSymbolValue("dentry_hashtable", &dentry_hashtable,
                      sizeof(dentry_hashtable));
  assert(rc == 0);
  rc = GetSymbolValue("d_hash_mask", &d_hash_mask, sizeof(d_hash_mask));
  assert(rc == 0);
  dentry_hashtable_limit = dentry_hashtable +
                           (d_hash_mask+1)*sizeof(struct list_head);

  /* Get address of dentry_unused, the head of the list threaded through
     the d_lru fields of the dentry */
  sP = LookupSymbolByName("dentry_unused");
  if (sP != NULL)
    dentry_unused = sP->addr;

  /* Compute bounds of dcache hash table */
  rc = GetSymbolValue("inode_hashtable", &inode_hashtable,
                      sizeof(inode_hashtable));
  assert(rc == 0);
  rc = GetSymbolValue("i_hash_mask", &i_hash_mask, sizeof(i_hash_mask));
  assert(rc == 0);
  inode_hashtable_limit = inode_hashtable +
                           (i_hash_mask+1)*sizeof(struct list_head);

  /* Get addresses of inode_in_use and inode_unused, the heads of the lists
     of inodes threaded through the i_list field */
  sP = LookupSymbolByName("inode_in_use");
  if (sP != NULL)
    inode_in_use = sP->addr;
  sP = LookupSymbolByName("inode_unused");
  if (sP != NULL)
    inode_unused = sP->addr;

  /* Get address of super_blocks, the head of the list of super_blocks
     threaded through the s_list field */
  sP = LookupSymbolByName("super_blocks");
  if (sP != NULL)
    super_blocks_addr = sP->addr;

  /* Get address of a task_struct on the list of all tasks */
#if 0
  rc = GetSymbolValue("init_tasks", &TaskAddress, sizeof(TaskAddress));
  assert(rc == 0);
#else
  sP = LookupSymbolByName("init_task_union");
  assert(sP != NULL);
  TaskAddress = sP->addr;
#endif

  /* Get address of the base of the array of struct pages */
  rc = GetSymbolValue("mem_map", &MemMapAddr, sizeof(MemMapAddr));
  assert(rc == 0);
}


/* Generic get routine for kernel objects.  Mallocs storage of the
   given size, then reads into the malloc'ed storage from the
   kernel address.  Frees the object and returns NULL if the address
   was invalid. */
void* GenericGet(unsigned long addr, int len)
{
  int rc;
  void* p;

  DBG(printf("GenericGet: addr 0x%lX len %d\n", addr, len));
  p = kMalloc(len);
  assert(p != NULL);
  rc = ReadKernel(addr, p, len);
  if (rc != 0)
  {
    kFree(p);
    return NULL;
  }
  return p;
}


/* Read a dentry from the kernel.  Space may need to be allocated for the
   name string. */
void* GetDentry(unsigned long addr)
{
  struct dentry* dP;
  char* nameP;
  int rc;

  dP = (struct dentry*) GenericGet(addr, sizeof(struct dentry));
  if (dP != NULL)
  {
    if (dP->d_name.len > 1024)
    {
      kFree(dP);
      return NULL;
    }
    else if (dP->d_name.len < DNAME_INLINE_LEN)
      dP->d_name.name = dP->d_iname;
    else
    {
      nameP = (char*) kMalloc(dP->d_name.len+1);
      assert(nameP != NULL);
      rc = ReadKernel((unsigned long)dP->d_name.name, nameP, dP->d_name.len+1);
      if (rc != 0)
      {
        kFree(nameP);
        kFree(dP);
        return NULL;
      }
      dP->d_name.name = nameP;
    }
  }
  return dP;
}


/* Free a dentry, including the name string */
void FreeDentry(void* p)
{
  struct dentry* dP = (struct dentry*) p;

  if (dP == NULL)
    return;
  if (dP->d_name.len >= DNAME_INLINE_LEN)
    kFree((void*)dP->d_name.name);
  kFree(dP);
}


/* Read an inode from the kernel */
void* GetInode(unsigned long addr)
{
  return GenericGet(addr, sizeof(struct inode));
}


/* Read a superblock from the kernel */
void* GetSuperBlock(unsigned long addr)
{
  return GenericGet(addr, sizeof(struct super_block));
}


/* Read a struct page from the kernel */
void* GetPage(unsigned long addr)
{
  return GenericGet(addr, sizeof(struct page));
}


/* Read an address_space from the kernel */
void* GetAddressSpace(unsigned long addr)
{
  return GenericGet(addr, sizeof(struct address_space));
}


/* Read a vm_area_struct from the kernel */
void* GetVMAreaStruct(unsigned long addr)
{
  return GenericGet(addr, sizeof(struct vm_area_struct));
}


/* Read an mm_struct from the kernel */
void* GetMMStruct(unsigned long addr)
{
  return GenericGet(addr, sizeof(struct mm_struct));
}


/* Read a task_struct from the kernel */
void* GetTaskStruct(unsigned long addr)
{
  return GenericGet(addr, sizeof(struct task_struct));
}


/* Read a task_struct from the kernel given its pid */
void* GetTaskStructByPID(unsigned long pid, unsigned long * addrP)
{
  struct task_struct * initP;
  struct task_struct * taskP;
  unsigned long nextTaskAddr;
  unsigned long initPid;

  initP = GetTaskStruct(TaskAddress);
  initPid = initP->pid;
  nextTaskAddr = (unsigned long)initP->next_task;
  CondFree(initP);

  for (;;)
  {
    taskP = GetTaskStruct(nextTaskAddr);
    if (taskP->pid == pid)
    {
      *addrP = nextTaskAddr;
      return taskP;
    }
    if (taskP->pid == initPid)
    {
      printf("PID %ld not found\n", pid);
      CondFree(taskP);
      return NULL;
    }
    nextTaskAddr = (unsigned long)taskP->next_task;
    CondFree(taskP);
  }
  return taskP;
}


#define P_HEADER(_ptr, _addr, _hdrtype, _hdrfield, _linktype, _linkfield) \
  if (_addr == ((unsigned long)(_ptr->_hdrfield.next) -                   \
                (unsigned long)(&(((struct _hdrtype*)0)->_hdrfield))))    \
    printf("  %-15s empty list\n", #_hdrfield);                           \
  else                                                                    \
  {                                                                       \
    struct _linktype* n =                                                 \
      list_entry(_ptr->_hdrfield.next, struct _linktype, _linkfield);     \
    struct _linktype* p =                                                 \
      list_entry(_ptr->_hdrfield.prev, struct _linktype, _linkfield);     \
    printf("  %-15s next: %s 0x%lX prev: %s 0x%lX\n",                     \
           #_hdrfield, #_linktype, n, #_linktype, p);                     \
  }


/* Print a dentry.  Parameter may be NULL. */
void PrintDentry(void* parm, unsigned long addr)
{
  struct dentry* dP = (struct dentry*) parm;
  unsigned long n;
  unsigned long p;
  unsigned long h;
  char nextBuf[256];
  char prevBuf[256];

  if (dP == NULL)
  {
    printf("NULL dentry\n");
    return;
  }

  printf("dentry 0x%lX:\n", addr);
  printf("  d_name          %s\n", dP->d_name.name);
  printf("  d_count         %d\n", atomic_read(&dP->d_count));
  printf("  d_flags         0x%X", dP->d_flags);
#if (LINUX_KERNEL_VERSION == 2040303) || (LINUX_KERNEL_VERSION == 2040312) && !NFS4_LINUX_2_4_4
  if (dP->d_flags & D_Autofs_Pending)  printf(" D_Autofs_Pending");
  if (dP->d_flags & D_NFS_Renamed)     printf(" D_NFS_Renamed");
  if (dP->d_flags & D_Disconnected)    printf(" D_Disconnected");
  if (dP->d_flags & D_Referenced)      printf(" D_Referenced");
#else
  if (dP->d_flags & DCACHE_AUTOFS_PENDING)    printf(" DCACHE_AUTOFS_PENDING");
  if (dP->d_flags & DCACHE_NFSFS_RENAMED)     printf(" DCACHE_NFSFS_RENAMED");
  if (dP->d_flags & DCACHE_NFSD_DISCONNECTED) printf(" DCACHE_NFSD_DISCONNECTED");
  if (dP->d_flags & DCACHE_REFERENCED)        printf(" DCACHE_REFERENCED");
#endif
  printf("\n");
  printf("  d_inode         inode 0x%lX\n", dP->d_inode);
  printf("  d_parent        dentry 0x%lX\n", dP->d_parent);
# if LINUX_KERNEL_VERSION < 2040600
# if 0
    P_HEADER(dP, addr, dentry, d_vfsmnt, vfsmount, mnt_clash);
# endif
# endif

  n = (unsigned long) dP->d_hash.next;
  if (addr == (unsigned long) list_entry(n, struct dentry, d_hash))
    printf("  d_hash          empty list\n");
  else
  {
    if (n >= dentry_hashtable  &&
        n < dentry_hashtable_limit)
      sprintf(nextBuf, "dentry_hashtable+0x%lX", n-dentry_hashtable);
    else
      sprintf(nextBuf, "dentry 0x%lX",
              list_entry(n, struct dentry, d_hash));
    p = (unsigned long) dP->d_hash.prev;
    if (p >= dentry_hashtable  &&
        p < dentry_hashtable_limit)
      sprintf(prevBuf, "dentry_hashtable+0x%lX", p-dentry_hashtable);
    else
      sprintf(prevBuf, "dentry 0x%lX",
              list_entry(p, struct dentry, d_hash));
    printf("  d_hash          next: %s prev: %s\n", nextBuf, prevBuf);
  }

  h = dentry_unused;
  n = (unsigned long) dP->d_lru.next;
  if (addr == (unsigned long) list_entry(n, struct dentry, d_lru))
    printf("  d_lru           empty list\n");
  else
  {
    if (n == h)
      sprintf(nextBuf, "dentry_unused head");
    else
      sprintf(nextBuf, "dentry 0x%lX",
              list_entry(n, struct dentry, d_lru));
    p = (unsigned long) dP->d_lru.prev;
    if (p == h)
      sprintf(prevBuf, "dentry_unused head");
    else
      sprintf(prevBuf, "dentry 0x%lX",
              list_entry(p, struct dentry, d_lru));
    printf("  d_lru           next: %s prev: %s\n", nextBuf, prevBuf);
  }

  if (dP->d_parent == NULL)
    h = (unsigned long) NULL;
  else
    h = (unsigned long) &dP->d_parent->d_subdirs;
  n = (unsigned long) dP->d_child.next;
  if (addr == (unsigned long) list_entry(n, struct dentry, d_child))
    printf("  d_child         empty list\n");
  else
  {
    if (n == h)
      sprintf(nextBuf, "dentry 0x%lX head",
              list_entry(n, struct dentry, d_subdirs));
    else
      sprintf(nextBuf, "dentry 0x%lX",
              list_entry(n, struct dentry, d_child));
    p = (unsigned long) dP->d_child.prev;
    if (p == h)
      sprintf(prevBuf, "dentry 0x%lX head",
              list_entry(p, struct dentry, d_subdirs));
    else
      sprintf(prevBuf, "dentry 0x%lX",
              list_entry(p, struct dentry, d_child));
    printf("  d_child         next: %s prev: %s\n", nextBuf, prevBuf);
  }

  P_HEADER(dP, addr, dentry, d_subdirs, dentry, d_child);

  h = (unsigned long) &dP->d_inode->i_dentry;
  n = (unsigned long) dP->d_alias.next;
  if (addr == (unsigned long) list_entry(n, struct dentry, d_alias))
    printf("  d_alias         empty list\n");
  else
  {
    if (n == h)
      sprintf(nextBuf, "inode 0x%lX", dP->d_inode);
    else
      sprintf(nextBuf, "dentry 0x%lX",
              list_entry(n, struct dentry, d_alias));
    p = (unsigned long) dP->d_alias.prev;
    if (p == h)
      sprintf(prevBuf, "inode 0x%lX", dP->d_inode);
    else
      sprintf(prevBuf, "dentry 0x%lX",
              list_entry(p, struct dentry, d_alias));
    printf("  d_alias         next: %s prev: %s\n", nextBuf, prevBuf);
  }

  printf("  d_time          0x%lX\n", dP->d_time);
  printf("  d_op            0x%lX %s\n",
         dP->d_op, ExactAddrToSymbol((unsigned long)dP->d_op));
  printf("  d_sb            super_block 0x%lX\n", dP->d_sb);
# if LINUX_KERNEL_VERSION >= 2040600 || NFS4_LINUX_2_4_4
    printf("  d_vfs_flags     0x%lX\n", dP->d_vfs_flags);
# endif
  printf("  d_fsdata        0x%lX\n", dP->d_fsdata);
  /* DumpMemory((char*)dP, sizeof(struct dentry), addr, DUMPMEM_I4); */
}


/* Print an inode.  Parameter may be NULL. */
void PrintInode(void* parm, unsigned long addr)
{
  struct inode* iP = (struct inode*) parm;
  unsigned long n;
  unsigned long p;
  unsigned long h;
  char nextBuf[256];
  char prevBuf[256];

  if (iP == NULL)
  {
    printf("NULL inode\n");
    return;
  }

  printf("inode 0x%lX:\n", addr);

  n = (unsigned long) iP->i_hash.next;
  if (addr == (unsigned long) list_entry(n, struct inode, i_hash))
    printf("  i_hash          empty list\n");
  else
  {
    if (n >= inode_hashtable  &&
        n < inode_hashtable_limit)
      sprintf(nextBuf, "inode_hashtable+0x%lX", n-inode_hashtable);
    else
      sprintf(nextBuf, "inode 0x%lX",
              list_entry(n, struct inode, i_hash));
    p = (unsigned long) iP->i_hash.prev;
    if (p >= inode_hashtable  &&
        p < inode_hashtable_limit)
      sprintf(prevBuf, "inode_hashtable+0x%lX", p-inode_hashtable);
    else
      sprintf(prevBuf, "inode 0x%lX",
              list_entry(p, struct inode, i_hash));
    printf("  i_hash          next: %s prev: %s\n", nextBuf, prevBuf);
  }

  n = (unsigned long) iP->i_list.next;
  if (addr == (unsigned long) list_entry(n, struct inode, i_list))
    printf("  i_list          empty list\n");
  else
  {
    if (n == inode_in_use)
      sprintf(nextBuf, "inode_in_use head");
    else if (n == inode_unused)
      sprintf(nextBuf, "inode_unused head");
    else
      sprintf(nextBuf, "inode 0x%lX",
              list_entry(n, struct inode, i_list));
    p = (unsigned long) iP->i_list.prev;
    if (p == inode_in_use)
      sprintf(prevBuf, "inode_in_use head");
    else if (p == inode_unused)
      sprintf(nextBuf, "inode_unused head");
    else
      sprintf(prevBuf, "inode 0x%lX",
              list_entry(p, struct inode, i_list));
    printf("  i_list          next: %s prev: %s\n", nextBuf, prevBuf);
  }

  P_HEADER(iP, addr, inode, i_dentry, dentry, d_alias);
  P_HEADER(iP, addr, inode, i_dirty_buffers, buffer_head, b_inode_buffers);
  printf("  i_ino           %d\n", iP->i_ino);
  printf("  i_count         %d\n", atomic_read(&iP->i_count));
  printf("  i_dev           0x%04X\n", iP->i_dev);
  printf("  i_mode          0x%08X\n", iP->i_mode);
  printf("  i_nlink         %d\n", iP->i_nlink);
  printf("  i_uid           %d\n", iP->i_uid);
  printf("  i_gid           %d\n", iP->i_gid);
  printf("  i_rdev          0x%04X\n", iP->i_rdev);
  printf("  i_size          %lld (0x%llX)\n", iP->i_size, iP->i_size);
  printf("  i_atime         %d\n", iP->i_atime);
  printf("  i_mtime         %d\n", iP->i_mtime);
  printf("  i_ctime         %d\n", iP->i_ctime);
  printf("  i_blksize       %ld\n", iP->i_blksize);
  printf("  i_blocks        %ld\n", iP->i_blocks);
  printf("  i_version       %ld\n", iP->i_version);
  /* i_sem */
  /* i_zombie */
  printf("  i_op            0x%lX %s\n",
         iP->i_op, ExactAddrToSymbol((unsigned long)iP->i_op));
  printf("  i_fop           0x%lX %s\n",
         iP->i_fop, ExactAddrToSymbol((unsigned long)iP->i_fop));
  printf("  i_sb            super_block 0x%lX\n", iP->i_sb);
  /* i_wait */
  if (iP->i_flock)
    printf("  i_flock         0x%lX\n", iP->i_flock);
  if (iP->i_mapping)
    printf("  i_mapping       address_space 0x%lX\n", iP->i_mapping);
  /* i_data */
  /* i_dquot */
  if (iP->i_pipe)
    printf("  i_pipe          0x%lX\n", iP->i_pipe);
  if (iP->i_bdev)
    printf("  i_bdev          0x%lX\n", iP->i_bdev);
  if (iP->i_dnotify_mask)
  printf("  i_dnotify_mask  0x%lX\n", iP->i_dnotify_mask);
  if (iP->i_dnotify)
    printf("  i_dnotify       0x%lX\n", iP->i_dnotify);
  printf("  i_state         0x%lX", iP->i_state);
  if (iP->i_state & I_DIRTY_SYNC)     printf(" I_DIRTY_SYNC");
  if (iP->i_state & I_DIRTY_DATASYNC) printf(" I_DIRTY_DATASYNC");
  if (iP->i_state & I_DIRTY_PAGES)    printf(" I_DIRTY_PAGES");
  if (iP->i_state & I_LOCK)           printf(" I_LOCK");
  if (iP->i_state & I_FREEING)        printf(" I_FREEING");
  if (iP->i_state & I_CLEAR)          printf(" I_CLEAR");
  printf("\n");
  printf("  i_flags         0x%lX", iP->i_flags);
  if (iP->i_flags & S_SYNC)           printf(" S_SYNC");
  if (iP->i_flags & S_NOATIME)        printf(" S_NOATIME");
  if (iP->i_flags & S_QUOTA)          printf(" S_QUOTA");
  if (iP->i_flags & S_APPEND)         printf(" S_APPEND");
  if (iP->i_flags & S_IMMUTABLE)      printf(" S_IMMUTABLE");
  if (iP->i_flags & S_DEAD)           printf(" S_DEAD");
  printf("\n");
  /* i_sock */
  if (atomic_read(&iP->i_writecount))
    printf("  i_writecount    %d\n", atomic_read(&iP->i_writecount));
  if (iP->i_attr_flags)
    printf("  i_attr_flags    0x%X\n", iP->i_attr_flags);
  printf("  i_generation    %d\n", iP->i_generation);
  printf("  generic_ip      cxiNode_t 0x%lX\n", iP->u.generic_ip);
  /* DumpMemory((char*)iP, sizeof(struct inode), addr, DUMPMEM_I4); */
}


/* Print a super_block.  Parameter may be NULL. */
void PrintSuperBlock(void* parm, unsigned long addr)
{
  struct super_block* sP = (struct super_block*) parm;
  unsigned long n;
  unsigned long p;
  unsigned long h;
  char nextBuf[256];
  char prevBuf[256];
  struct file_system_type* fstypeP;
  char* fstypeNameP;

  if (sP == NULL)
  {
    printf("NULL super_block\n");
    return;
  }

  printf("super_block 0x%lX:\n", addr);

  n = (unsigned long) sP->s_list.next;
  if (addr == (unsigned long) list_entry(n, struct super_block, s_list))
    printf("  s_list          empty list\n");
  else
  {
    if (n == super_blocks_addr)
      sprintf(nextBuf, "super_blocks head");
    else
      sprintf(nextBuf, "super_block 0x%lX",
              list_entry(n, struct super_block, s_list));
    p = (unsigned long) sP->s_list.prev;
    if (p == super_blocks_addr)
      sprintf(prevBuf, "super_blocks head");
    else
      sprintf(prevBuf, "super_block 0x%lX",
              list_entry(p, struct super_block, s_list));
    printf("  s_list          next: %s prev: %s\n", nextBuf, prevBuf);
  }

  printf("  s_dev           0x%04X\n", sP->s_dev);
  printf("  s_blocksize     %d\n", sP->s_blocksize);
  printf("  s_blocksize_bits %d\n", sP->s_blocksize_bits);
  printf("  s_lock          %d\n", sP->s_lock);
  printf("  s_dirt          %d\n", sP->s_dirt);

  printf("  s_type          0x%lX\n", sP->s_type);
  if (sP->s_type != NULL)
  {
    fstypeP = (struct file_system_type*) GenericGet((unsigned long) sP->s_type,
                                                    sizeof(struct file_system_type));
    if (fstypeP != NULL)
    {
      fstypeNameP = GenericGet((unsigned long) fstypeP->name, 256);
      if (fstypeNameP != NULL)
      {
        fstypeNameP[255] = '\0';
        printf("    name            %s\n", fstypeNameP);
        kFree(fstypeNameP);
      }
      printf("    fs_flags        0x%X\n", fstypeP->fs_flags);
      printf("    read_super      0x%X %s\n",
             fstypeP->read_super,
             ExactAddrToSymbol((unsigned long)fstypeP->read_super));
      printf("    owner           0x%X\n", fstypeP->owner);
      kFree(fstypeP);
    }
  }

  printf("  s_op            0x%lX %s\n",
         sP->s_op, ExactAddrToSymbol((unsigned long)sP->s_op));
  if (sP->dq_op)
    printf("  dq_op           0x%lX %s\n",
           sP->dq_op, ExactAddrToSymbol((unsigned long)sP->dq_op));
  printf("  s_flags         0x%lX\n", sP->s_flags);
  printf("  s_magic         0x%lX\n", sP->s_magic);
  printf("  s_root          dentry 0x%lX\n", sP->s_root);
  /* s_wait */
  P_HEADER(sP, addr, super_block, s_dirty, inode, i_list);
  P_HEADER(sP, addr, super_block, s_files, file, f_list);
  printf("  s_bdev          0x%lX\n", sP->s_bdev);
#if LINUX_KERNEL_VERSION < 2040900
#if 0
  P_HEADER(sP, addr, super_block, s_mounts, vfsmount, mnt_instances);
#endif
#endif
  printf("  generic_sbp     0x%lX\n", sP->u.generic_sbp);
}


/* Print a page struct.  Parameter may be NULL. */
void PrintPage(void* parm, unsigned long addr)
{
  struct page* pageP = (struct page*) parm;

  if (pageP == NULL)
  {
    printf("NULL page\n");
    return;
  }

  printf("page 0x%lX (frame number 0x%lX):\n",
         addr, (addr-MemMapAddr)/sizeof(struct page));

  P_HEADER(pageP, addr, page, list, page, list);
  printf("  mapping         address_space 0x%lX\n", pageP->mapping);
  printf("  index           0x%lX\n", pageP->index);
  printf("  next_hash       page 0x%lX\n", pageP->next_hash);
  printf("  count           %d\n", atomic_read(&pageP->count));
  printf("  flags           0x%lX", pageP->flags);
  if (pageP->flags & (1 << PG_locked))          printf(" locked");
  if (pageP->flags & (1 << PG_error))           printf(" error");
  if (pageP->flags & (1 << PG_referenced))      printf(" referenced");
  if (pageP->flags & (1 << PG_uptodate))        printf(" uptodate");
  if (pageP->flags & (1 << PG_dirty))           printf(" dirty");
#ifdef PG_decr_after
  if (pageP->flags & (1 << PG_decr_after))      printf(" decr_after");
#endif
  if (pageP->flags & (1 << PG_active))          printf(" active");
  if (pageP->flags & (1 << PG_slab))            printf(" slab");
#ifdef PG_inactive_dirty
  if (pageP->flags & (1 << PG_inactive_dirty))  printf(" inactive_dirty");
#endif
#ifdef PG_inactive_clean
  if (pageP->flags & (1 << PG_inactive_clean))  printf(" inactive_clean");
#endif
#ifdef PG_swap_cache
  if (pageP->flags & (1 << PG_swap_cache))      printf(" swap_cache");
#endif
#ifdef PG_checked
  if (pageP->flags & (1 << PG_checked))         printf(" checked");
#endif
  if (pageP->flags & (1 << PG_skip))            printf(" skip");
  if (pageP->flags & (1 << PG_highmem))         printf(" highmem");
  if (pageP->flags & (1 << PG_arch_1))          printf(" arch_1");
  printf("\n");
  P_HEADER(pageP, addr, page, lru, page, lru);
#if LINUX_KERNEL_VERSION < 2041000
  printf("  age             %d\n", pageP->age);
#endif
  /* wait */
  printf("  pprev_hash      0x%lX\n", pageP->pprev_hash);
  printf("  buffers         0x%lX\n", pageP->buffers);
#ifdef CONFIG_HIGHMEM
  printf("  virtual         0x%lX\n", pageP->virtual);
#endif
#if LINUX_KERNEL_VERSION < 2041804
  printf("  zone            0x%lX\n", pageP->zone);
#endif
}


/* Print an address_space struct.  Parameter may be NULL. */
void PrintAddressSpace(void* parm, unsigned long addr)
{
  struct address_space* asP = (struct address_space*) parm;

  if (asP == NULL)
  {
    printf("NULL address_space\n");
    return;
  }

  printf("address_space 0x%lX:\n", addr);

  P_HEADER(asP, addr, address_space, clean_pages, page, list);
  P_HEADER(asP, addr, address_space, dirty_pages, page, list);
  P_HEADER(asP, addr, address_space, locked_pages, page, list);
  printf("  nrpages         %d\n", asP->nrpages);
  /* page_lock */
  printf("  a_ops           0x%lX %s\n",
         asP->a_ops, ExactAddrToSymbol((unsigned long)asP->a_ops));
  printf("  host            inode 0x%lX\n", asP->host);
  printf("  i_mmap          vm_area_struct 0x%lX\n", asP->i_mmap);
  printf("  i_mmap_shared   vm_area_struct 0x%lX\n", asP->i_mmap_shared);
  /* i_shared_lock */
  printf("  gfp_mask        0x%X\n", asP->gfp_mask);
}


/* Print a vm_area_struct struct.  Parameter may be NULL. */
void PrintVMAreaStruct(void* parm, unsigned long addr)
{
  struct vm_area_struct* vmP = (struct vm_area_struct*) parm;

  if (vmP == NULL)
  {
    printf("NULL vm_area_struct\n");
    return;
  }

  printf("vm_area_struct 0x%lX:\n", addr);

  printf("  vm_mm           mm_struct 0x%lX\n", vmP->vm_mm);
  printf("  vm_start        0x%lX\n", vmP->vm_start);
  printf("  vm_end          0x%lX\n", vmP->vm_end);
  printf("  vm_next         vm_area_struct 0x%lX\n", vmP->vm_next);
  printf("  vm_page_prot    0x%lX\n", vmP->vm_page_prot.pgprot);
  printf("  vm_flags        0x%lX", vmP->vm_flags);
  if (vmP->vm_flags & VM_READ)        printf(" VM_READ");
  if (vmP->vm_flags & VM_WRITE)       printf(" VM_WRITE");
  if (vmP->vm_flags & VM_EXEC)        printf(" VM_EXEC");
  if (vmP->vm_flags & VM_SHARED)      printf(" VM_SHARED");
  if (vmP->vm_flags & VM_MAYREAD)     printf(" VM_MAYREAD");
  if (vmP->vm_flags & VM_MAYWRITE)    printf(" VM_MAYWRITE");
  if (vmP->vm_flags & VM_MAYEXEC)     printf(" VM_MAYEXEC");
  if (vmP->vm_flags & VM_MAYSHARE)    printf(" VM_MAYSHARE");
  if (vmP->vm_flags & VM_GROWSDOWN)   printf(" VM_GROWSDOWN");
  if (vmP->vm_flags & VM_GROWSUP)     printf(" VM_GROWSUP");
  if (vmP->vm_flags & VM_SHM)         printf(" VM_SHM");
  if (vmP->vm_flags & VM_DENYWRITE)   printf(" VM_DENYWRITE");
  if (vmP->vm_flags & VM_EXECUTABLE)  printf(" VM_EXECUTABLE");
  if (vmP->vm_flags & VM_LOCKED)      printf(" VM_LOCKED");
  if (vmP->vm_flags & VM_IO)          printf(" VM_IO");
  if (vmP->vm_flags & VM_SEQ_READ)    printf(" VM_SEQ_READ");
  if (vmP->vm_flags & VM_RAND_READ)   printf(" VM_RAND_READ");
  if (vmP->vm_flags & VM_DONTCOPY)    printf(" VM_DONTCOPY");
  if (vmP->vm_flags & VM_DONTEXPAND)  printf(" VM_DONTEXPAND");
  if (vmP->vm_flags & VM_RESERVED)    printf(" VM_RESERVED");
  printf("\n");
#if LINUX_KERNEL_VERSION < 2041000
  printf("  vm_avl_height   %d\n", vmP->vm_avl_height);
  printf("  vm_avl_left     vm_area_struct 0x%lX\n", vmP->vm_avl_left);
  printf("  vm_avl_right    vm_area_struct 0x%lX\n", vmP->vm_avl_right);
#endif
  printf("  vm_next_share   vm_area_struct 0x%lX\n", vmP->vm_next_share);
  /* vm_pprev_share */
  printf("  vm_ops          0x%lX %s\n",
         vmP->vm_ops, ExactAddrToSymbol((unsigned long)vmP->vm_ops));
  printf("  vm_pgoff        0x%lX\n", vmP->vm_pgoff);
  printf("  vm_file         0x%lX\n", vmP->vm_file);
  printf("  vm_raend        0x%lX\n", vmP->vm_raend);
  printf("  vm_private_data 0x%lX\n", vmP->vm_private_data);
}


/* Print an mm_struct struct.  Parameter may be NULL. */
void PrintMMStruct(void* parm, unsigned long addr)
{
  struct mm_struct* mmP = (struct mm_struct*) parm;

  if (mmP == NULL)
  {
    printf("NULL mm_struct\n");
    return;
  }

  printf("mm_struct 0x%lX:\n", addr);

  printf("  mmap            vm_area_struct 0x%lX\n", mmP->mmap);
#if LINUX_KERNEL_VERSION < 2041000
  printf("  mmap_avl        vm_area_struct 0x%lX\n", mmP->mmap_avl);
#endif
  printf("  mmap_cache      vm_area_struct 0x%lX\n", mmP->mmap_cache);
  printf("  pgd             pgd 0x%lX\n", mmP->pgd);
  printf("  mm_users        %d\n", atomic_read(&mmP->mm_users));
  printf("  mm_count        %d\n", atomic_read(&mmP->mm_count));
  printf("  map_count       %d\n", mmP->map_count);
  /* mmap_sem */
  /* page_table_lock */
  P_HEADER(mmP, addr, mm_struct, mmlist, mm_struct, mmlist);
  printf("  start_code      0x%lX\n", mmP->start_code);
  printf("  end_code        0x%lX\n", mmP->end_code);
  printf("  start_data      0x%lX\n", mmP->start_data);
  printf("  end_data        0x%lX\n", mmP->end_data);
  printf("  start_brk       0x%lX\n", mmP->start_brk);
  printf("  brk             0x%lX\n", mmP->brk);
  printf("  start_stack     0x%lX\n", mmP->start_stack);
  printf("  arg_start       0x%lX\n", mmP->arg_start);
  printf("  arg_end         0x%lX\n", mmP->arg_end);
  printf("  env_start       0x%lX\n", mmP->env_start);
  printf("  env_end         0x%lX\n", mmP->env_end);
  printf("  rss             %lu\n", mmP->rss);
  printf("  total_vm        %lu\n", mmP->total_vm);
  printf("  locked_vm       %lu\n", mmP->locked_vm);
  printf("  def_flags       0x%lX\n", mmP->def_flags);
  printf("  cpu_vm_mask     0x%lX\n", mmP->cpu_vm_mask);
#if LINUX_KERNEL_VERSION < 2041800
  printf("  swap_address    0x%lX\n", mmP->swap_address);
#endif
}


/* Print an task_struct struct.  Parameter may be NULL. */
void PrintTaskStruct(void* parm, unsigned long addr)
{
  struct task_struct* taskP = (struct task_struct*) parm;

  if (taskP == NULL)
  {
    printf("NULL task_struct\n");
    return;
  }

  printf("task_struct 0x%lX:\n", addr);

  printf("  state           0x%lX\n", taskP->state);
  printf("  flags           0x%lX\n", taskP->flags);
  printf("  mm              mm_struct 0x%lX\n", taskP->mm);
  printf("  next_task       task_struct 0x%lX\n", taskP->next_task);
  printf("  prev_task       task_struct 0x%lX\n", taskP->prev_task);
  printf("  active_mm       mm_struct 0x%lX\n", taskP->active_mm);
  printf("  pid             %d\n", taskP->pid);
  printf("  user            0x%lX\n", taskP->user);
  printf("  comm            %s\n", taskP->comm);
  printf("  files           0x%lX\n", taskP->files);
}


#ifdef GPFS_ARCH_I386
/* Display page table flag bits */
static void ShowFlagBits(unsigned long f)
{
  if (f & _PAGE_PRESENT)  printf(" present");
  if (f & _PAGE_RW)       printf(" RW");
  if (f & _PAGE_USER)     printf(" user");
  if (f & _PAGE_PWT)      printf(" PWT");
  if (f & _PAGE_PCD)      printf(" PCD");
  if (f & _PAGE_ACCESSED) printf(" accessed");
  if (f & _PAGE_DIRTY)    printf(" dirty");
  if (f & _PAGE_PSE)      printf(" 4M");
  if (f & _PAGE_GLOBAL)   printf(" global");
  printf("\n");
}


/* Translate vaddr using pgd at addr */
void VirtToReal(unsigned long pgdAddr, unsigned long vaddr)
{
  pgd_t* pgdP;
  unsigned long pgdIndex;
  pgd_t pgdEntry;
  unsigned long pgdVal;

  unsigned long pmdAddr;
  pmd_t* pmdP;
  pmd_t pmdEntry;
  unsigned long pmdVal;
  unsigned long pageNumber;

  unsigned long pteAddr;
  unsigned long pteIndex;
  pte_t* pteP;
  pte_t pteEntry;
  unsigned long pteVal;

#ifdef CONFIG_X86_PAE
  printf("Could not read pmd\n");
  return;
#else
  pgdIndex = pgd_index(vaddr);
  pgdAddr = (unsigned long)(((pgd_t*)pgdAddr) + pgdIndex);
  pgdP = (pgd_t*)GenericGet(pgdAddr, sizeof(*pgdP));
  if (pgdP == NULL)
  {
    printf("Could not read pgd\n");
    return;
  }
  pgdEntry = *pgdP;
  CondFree(pgdP);
  pgdVal = pgd_val(pgdEntry);
  printf("pgdAddr 0x%lX pgd entry 0x%lX", pgdAddr, pgdVal);
  ShowFlagBits(pgdVal);
  if (!(pgdVal & _PAGE_PRESENT))
  {
    printf("PGD not present\n");
    return;
  }

  pmdAddr = (unsigned long) pmd_offset((pgd_t*)pgdAddr, vaddr);
  /* Above only works for two level page tables, because the code for
     3-level tables actually dereferences pgdAddr, which we cannot do
     from this user-level program. */
  pmdP = (pmd_t*)GenericGet(pmdAddr, sizeof(*pmdP));
  if (pmdP == NULL)
  {
    printf("Could not read pmd\n");
    return;
  }
  pmdEntry = *pmdP;
  CondFree(pmdP);
  pmdVal = pmd_val(pmdEntry);
  printf("pmdAddr 0x%lX pmd entry 0x%lX", pmdAddr, pmdVal);
  ShowFlagBits(pmdVal);
  if (!(pmdVal & _PAGE_PRESENT))
  {
    printf("PMD not present\n");
    return;
  }

  if (pmdVal & _PAGE_PSE)
  {
    pageNumber = (pgdVal>>PAGE_SHIFT) + ((vaddr & (PGDIR_SIZE-1))>>PAGE_SHIFT);
    printf("vaddr 0x%lX pageNumber 0x%lX page 0x%lX\n",
           vaddr, pageNumber, MemMapAddr + pageNumber*sizeof(struct page));
    return;
  }

  pteAddr = (unsigned long) __va(pmdVal & PAGE_MASK);
  pteIndex = __pte_offset(vaddr);
  pteAddr = (unsigned long)(((pte_t*)pteAddr) + pteIndex);
  pteP = (pte_t*)GenericGet(pteAddr, sizeof(pte_t));
  pteEntry = *pteP;
  CondFree(pteP);
  pteVal = pte_val(pteEntry);
  printf("pteAddr 0x%lX pte entry 0x%lX", pteAddr, pteVal);
  ShowFlagBits(pteVal);
  if (!pte_present(pteEntry))
  {
    printf("PTE not present\n");
    return;
  }

  pageNumber = pteVal >> PAGE_SHIFT;
  printf("vaddr 0x%lX pageNumber 0x%lX page 0x%lX\n",
         vaddr, pageNumber, MemMapAddr + pageNumber*sizeof(struct page));
#endif
}
#endif  /* GPFS_ARCH_I386 */

